package presentation;

/**
 * @author Alberto Iachini
 */

import java.awt.GridBagLayout;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JButton;
import java.awt.GridBagConstraints;
import javax.swing.JLabel;
import java.awt.Insets;
import java.io.File;
import presentation.graph.GraphFactory;
import presentation.util.Exporter;

import org.jgraph.JGraph;
import javax.swing.JScrollPane;
import javax.swing.filechooser.FileNameExtensionFilter;

import abstraction.Service;

/**
 * This class is the panel used to show the cartesian product of all available services
 */
public class CartesianProductPanel extends JPanel {

	private static final long serialVersionUID = 971167226552503028L;
	
	private OrchestratorFrame owner = null;

	private JButton changeRepresentationButton = null;
	private JButton saveToImageButton = null;
	private JButton saveToDotButton = null;
	private JButton showProductButton = null;
	
	private JLabel cartesianProductLabel = null;
	private JLabel statusLabel = null;
	
	private JScrollPane cartesianProductScrollPane = null;
	
	private JGraph cartesianProductGraph = null;
	private Service cartesianProduct = null; 
	

	/**
	 * This is the default constructor
	 */
	public CartesianProductPanel(OrchestratorFrame frame) {
		super();
		initialize();
		owner = frame;
	}

	/**
	 * This method initializes this
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints12 = new GridBagConstraints();
		gridBagConstraints12.gridx = 0;
		gridBagConstraints12.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints12.ipadx = 14;
		gridBagConstraints12.gridy = 4;
		GridBagConstraints gridBagConstraints11 = new GridBagConstraints();
		gridBagConstraints11.gridx = 0;
		gridBagConstraints11.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints11.ipadx = 52;
		gridBagConstraints11.gridy = 1;
		GridBagConstraints gridBagConstraints5 = new GridBagConstraints();
		gridBagConstraints5.fill = GridBagConstraints.BOTH;
		gridBagConstraints5.gridy = 1;
		gridBagConstraints5.weightx = 10.0;
		gridBagConstraints5.weighty = 10.0;
		gridBagConstraints5.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints5.gridheight = 6;
		gridBagConstraints5.gridx = 1;
		GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
		gridBagConstraints3.gridx = 1;
		gridBagConstraints3.anchor = GridBagConstraints.WEST;
		gridBagConstraints3.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints3.fill = GridBagConstraints.HORIZONTAL;
		gridBagConstraints3.gridy = 0;
		statusLabel = new JLabel();
		statusLabel.setText("Cartesian Product");
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.anchor = GridBagConstraints.WEST;
		gridBagConstraints2.ipadx = 38;
		gridBagConstraints2.gridy = 3;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.anchor = GridBagConstraints.WEST;
		gridBagConstraints1.ipadx = 43;
		gridBagConstraints1.gridy = 0;
		cartesianProductLabel = new JLabel();
		cartesianProductLabel.setText("Cartesian Product");
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.anchor = GridBagConstraints.WEST;
		gridBagConstraints.ipadx = 49;
		gridBagConstraints.gridy = 2;
		this.setSize(300, 200);
		this.setLayout(new GridBagLayout());
		this.add(getSaveToImageButton(), gridBagConstraints);
		this.add(cartesianProductLabel, gridBagConstraints1);
		this.add(getSaveToDotButton(), gridBagConstraints2);
		this.add(statusLabel, gridBagConstraints3);
		this.add(getCartesianProductScrollPane(), gridBagConstraints5);
		this.add(getShowProductButton(), gridBagConstraints11);
		this.add(getChangeRepresentationButton(), gridBagConstraints12);
	}

	/**
	 * This method initializes saveToImageButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToImageButton() {
		if (saveToImageButton == null) {
			saveToImageButton = new JButton();
			saveToImageButton.setText("Save To Image");
			saveToImageButton.setToolTipText("Save the graph in a jpeg file");
			saveToImageButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					if(cartesianProductGraph != null)
					{
						File file = saveImageFile();
						if(file != null)
							Exporter.exportToImage(file, cartesianProductGraph);
					}
				}
			});
		}
		return saveToImageButton;
	}

	/**
	 * This method initializes saveToDotButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToDotButton() {
		if (saveToDotButton == null) {
			saveToDotButton = new JButton();
			saveToDotButton.setText("Save To DOT File");
			saveToDotButton.setToolTipText("Save the graph in DOT format");
			saveToDotButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					if(cartesianProduct != null)
					{
						File file = saveDotFile();
						if(file != null)
							Exporter.exportToDot(file, cartesianProduct);
					}
				}
			});
		}
		return saveToDotButton;
	}


	/**
	 * This method initializes orchestratorScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getCartesianProductScrollPane() {
		if (cartesianProductScrollPane == null) {
			cartesianProductScrollPane = new JScrollPane();
			cartesianProductScrollPane.setViewportView(null);
		}
		return cartesianProductScrollPane;
	}

	/**
	 * This method initializes showProductButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getShowProductButton() {
		if (showProductButton == null) {
			showProductButton = new JButton();
			showProductButton.setText("Show Product");
			showProductButton.setToolTipText("Show the cartesian product");
			showProductButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					/*
					 * shows the cartesian product
					 * 
					 */	
					if(!owner.isAvailableLoaded())
					{
						JOptionPane.showMessageDialog(owner, "No available services loades", "Warning", JOptionPane.WARNING_MESSAGE);
					}
					else
					{
						int opt = JOptionPane.showConfirmDialog(owner, "Expensive operation","Warning",
								JOptionPane.YES_NO_OPTION,
								JOptionPane.QUESTION_MESSAGE);
						if(opt == JOptionPane.YES_OPTION)
						{
							long time = System.currentTimeMillis();
							cartesianProduct = owner.getProduct();
							cartesianProductGraph = (new GraphFactory(cartesianProduct)).getGraph();
							cartesianProductScrollPane.setViewportView(cartesianProductGraph);
							time = System.currentTimeMillis()-time;
							
							long ms = time%1000;
							time = time/1000;
							long sec = time%60;
							time = time/60;
							long min = time%60;
							time = time/60;
							
							String mess = "Elapsed Time: "+ time+ " hours, "
								+ min + " minutes, " 
								+ sec + " seconds, "
								+ ms + " milliseconds";
							JOptionPane.showMessageDialog(owner, mess, "Time", JOptionPane.INFORMATION_MESSAGE);
						}
					}
				}
			});
		}
		return showProductButton;
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into jpeg file
	 * 
	 * @return the selected file
	 */
	private File saveImageFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		FileNameExtensionFilter filter = new FileNameExtensionFilter("JPEG File", "jpeg");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into dot file
	 * 
	 * @return the selected file
	 */
	private File saveDotFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		FileNameExtensionFilter filter = new FileNameExtensionFilter("DOT File", "dot");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}

	/**
	 * This method initializes changeRepresentationButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getChangeRepresentationButton() {
		if (changeRepresentationButton == null) {
			changeRepresentationButton = new JButton();
			changeRepresentationButton.setText("Change Visualization");
			changeRepresentationButton.setToolTipText("Change the layout of the graph");
			changeRepresentationButton
					.addActionListener(new java.awt.event.ActionListener() {
						public void actionPerformed(java.awt.event.ActionEvent e) {
							if(cartesianProductGraph != null)
							{
								ChoiceLayoutDialog dialog = new ChoiceLayoutDialog(owner);
								String layout = dialog.getGraphLayout();
								if(layout != null)
								{
									GraphFactory.changeRepresentation(cartesianProductGraph, layout);
								}
							}
						}
					});
		}
		return changeRepresentationButton;
	}

}
